<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="csrf-token" content="{{ csrf_token() }}">
    <title>Wireless Link Calculator</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/chartjs-plugin-annotation@1.0.2/dist/chartjs-plugin-annotation.min.js"></script>
    <script src="https://maps.googleapis.com/maps/api/js?key=AIzaSyDe_VK5Xg2UvPgqfMke10cN6SGXgySml8Y&callback=initMap" async defer></script>
    <style>
        #profileCanvas {
            width: 100%;
            height: 500px;
            background-color: #f8fafc;
        }
        .antenna-icon {
            width: 24px;
            height: 24px;
            background-image: url('data:image/svg+xml;utf8,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="%233B82F6"><path d="M12 3l-6 6h3v11h2v-7h2v7h2V9h3l-6-6z"/></svg>');
            background-repeat: no-repeat;
            position: absolute;
            z-index: 100;
        }
        #map {
            height: 400px;
        }
        .warning-banner {
            background-color: #FEE2E2;
            border-left: 4px solid #EF4444;
            padding: 1rem;
            margin-bottom: 1rem;
        }
        .legend {
            display: flex;
            flex-wrap: wrap;
            gap: 1rem;
            margin-top: 1rem;
        }
        .legend-item {
            display: flex;
            align-items: center;
            font-size: 0.875rem;
        }
        .legend-color {
            width: 20px;
            height: 20px;
            margin-right: 0.5rem;
            border-radius: 3px;
        }
        .loading-overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(255,255,255,0.8);
            display: flex;
            justify-content: center;
            align-items: center;
            z-index: 1000;
        }
    </style>
</head>
<body class="bg-gray-100 min-h-screen p-4">
    <div id="loadingOverlay" class="loading-overlay">
        <div class="text-center">
            <div class="animate-spin rounded-full h-12 w-12 border-t-2 border-b-2 border-blue-500 mx-auto"></div>
            <p class="mt-4 text-gray-700">Loading application...</p>
        </div>
    </div>

    <div class="max-w-6xl mx-auto bg-white rounded-lg shadow-lg p-6">
        <h1 class="text-2xl font-bold text-gray-800 mb-6">Wireless Link Calculator</h1>
        
        <div id="warningBanner" class="warning-banner hidden">
            <div class="flex items-center text-red-600">
                <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"></path>
                </svg>
                <span class="font-medium">No direct visibility due to obstacles</span>
            </div>
        </div>
        
        <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
            <!-- Input Form Column -->
            <div class="lg:col-span-1 bg-gray-50 p-4 rounded-lg">
                <h2 class="text-xl font-semibold mb-4">Link Parameters</h2>
                <div class="space-y-4">
                    <!-- Transmitter Section -->
                    <div class="border-b pb-4">
                        <h3 class="font-medium text-gray-700 mb-2">Transmitter</h3>
                        <div class="space-y-3">
                            <div>
                                <label class="block text-sm font-medium text-gray-700">Power (dBm)</label>
                                <input id="tx_power" type="number" step="0.1" value="20" class="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700">Antenna Gain (dBi)</label>
                                <input id="tx_antenna_gain" type="number" step="0.1" value="10" class="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700">Height Above Ground (m)</label>
                                <input id="start_height" type="number" step="0.1" value="10" class="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3">
                            </div>
                        </div>
                    </div>

                    <!-- Receiver Section -->
                    <div class="border-b pb-4">
                        <h3 class="font-medium text-gray-700 mb-2">Receiver</h3>
                        <div class="space-y-3">
                            <div>
                                <label class="block text-sm font-medium text-gray-700">Antenna Gain (dBi)</label>
                                <input id="rx_antenna_gain" type="number" step="0.1" value="10" class="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700">Height Above Ground (m)</label>
                                <input id="end_height" type="number" step="0.1" value="10" class="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3">
                            </div>
                        </div>
                    </div>

                    <!-- RF Parameters -->
                    <div class="border-b pb-4">
                        <h3 class="font-medium text-gray-700 mb-2">RF Parameters</h3>
                        <div>
                            <label class="block text-sm font-medium text-gray-700">Frequency (GHz)</label>
                            <input id="frequency" type="number" step="0.01" value="2.4" class="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3">
                        </div>
                    </div>

                    <!-- Location Parameters -->
                    <div>
                        <h3 class="font-medium text-gray-700 mb-2">Locations</h3>
                        <div class="grid grid-cols-2 gap-3">
                            <div>
                                <label class="block text-sm font-medium text-gray-700">Start Latitude</label>
                                <input id="start_lat" type="number" step="0.000001" value="37.7749" class="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700">Start Longitude</label>
                                <input id="start_lon" type="number" step="0.000001" value="-122.4194" class="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700">End Latitude</label>
                                <input id="end_lat" type="number" step="0.000001" value="37.7849" class="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700">End Longitude</label>
                                <input id="end_lon" type="number" step="0.000001" value="-122.4094" class="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3">
                            </div>
                        </div>
                    </div>

                    <button onclick="calculateLinkBudget()" class="w-full bg-blue-600 hover:bg-blue-700 text-white font-medium py-2 px-4 rounded-md">
                        Calculate Link Budget
                    </button>
                </div>
            </div>
            
            <!-- Visualization Column -->
            <div class="lg:col-span-2 space-y-6">
                <!-- Map Container -->
                <div class="border rounded-lg overflow-hidden">
                    <div id="map" class="w-full"></div>
                    <div class="p-4 bg-gray-50">
                        <p id="mapInfo" class="text-sm text-gray-600">Click on the elevation profile or drag markers to update locations</p>
                    </div>
                </div>
                
                <!-- Profile Visualization -->
                <div class="border rounded-lg overflow-hidden">
                    <div class="p-4 bg-gray-50 border-b">
                        <h2 class="font-semibold text-gray-800">Link Profile Analysis</h2>
                    </div>
                    <div class="p-4 relative">
                        <canvas id="profileCanvas"></canvas>
                    </div>
                    <div class="p-4 bg-gray-50 border-t">
                        <div id="losStatus" class="text-sm">
                            <p>Line of sight analysis will appear here after calculation</p>
                        </div>
                        <div class="legend">
                            <div class="legend-item">
                                <div class="legend-color" style="background-color: #10B981;"></div>
                                <span>Terrain Elevation</span>
                            </div>
                            <div class="legend-item">
                                <div class="legend-color" style="background-color: #3B82F6;"></div>
                                <span>LOS Path</span>
                            </div>
                            <div class="legend-item">
                                <div class="legend-color" style="background-color: rgba(239, 68, 68, 0.2);"></div>
                                <span>60% Fresnel Zone</span>
                            </div>
                            <div class="legend-item">
                                <div class="legend-color" style="background-color: rgba(239, 68, 68, 0.1);"></div>
                                <span>Full Fresnel Zone</span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        // Global variables
        let map, profileChart;
        const elevationPoints = [];
        const distancePoints = [];
        let markerStart, markerEnd, polyline;
        let obstructionAnnotation = null;
        let mapInfoElement = null;
        let highlightMarker = null;

        // Initialize Google Map
        function initMap() {
            try {
                // Default coordinates
                const defaultStart = { lat: 37.7749, lng: -122.4194 };
                const defaultEnd = { lat: 37.7849, lng: -122.4094 };
                
                // Initialize map
                map = new google.maps.Map(document.getElementById("map"), {
                    center: defaultStart,
                    zoom: 12,
                });

                // Initialize map info element
                mapInfoElement = document.getElementById('mapInfo');

                // Create markers with default positions
                markerStart = new google.maps.Marker({
                    position: defaultStart,
                    map: map,
                    title: "TX Antenna",
                    icon: {
                        url: "http://maps.google.com/mapfiles/ms/icons/green-dot.png"
                    },
                    draggable: true
                });

                markerEnd = new google.maps.Marker({
                    position: defaultEnd,
                    map: map,
                    title: "RX Antenna",
                    icon: {
                        url: "http://maps.google.com/mapfiles/ms/icons/red-dot.png"
                    },
                    draggable: true
                });

                // Update coordinates when markers are dragged
                google.maps.event.addListener(markerStart, 'dragend', updateCoordinates);
                google.maps.event.addListener(markerEnd, 'dragend', updateCoordinates);

                // Initialize path
                updateMapPath();
                initProfileChart();
                
                // Hide loading overlay
                document.getElementById('loadingOverlay').style.display = 'none';
            } catch (error) {
                console.error("Error initializing map:", error);
                document.getElementById('loadingOverlay').innerHTML = `
                    <div class="text-center">
                        <div class="text-red-500 mb-4">
                            <svg class="w-12 h-12 mx-auto" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"></path>
                            </svg>
                        </div>
                        <p class="text-red-600 font-medium">Failed to load map. Please refresh the page.</p>
                        <p class="text-gray-600 mt-2">${error.message}</p>
                    </div>
                `;
            }
        }

        // Update the path between points
        function updateMapPath() {
            try {
                if (!map || !markerStart || !markerEnd) return;
                
                // Remove previous polyline if it exists
                if (polyline) {
                    polyline.setMap(null);
                }
                
                // Get current marker positions
                const startPos = markerStart.getPosition();
                const endPos = markerEnd.getPosition();
                
                if (!startPos || !endPos) return;
                
                // Create new polyline
                polyline = new google.maps.Polyline({
                    path: [startPos, endPos],
                    geodesic: true,
                    strokeColor: "#3B82F6",
                    strokeOpacity: 0.5,
                    strokeWeight: 2,
                    map: map
                });
            } catch (error) {
                console.error("Error updating map path:", error);
            }
        }

        // Update coordinates when markers are dragged
        function updateCoordinates() {
            try {
                if (!markerStart || !markerEnd) return;
                
                const startPos = markerStart.getPosition();
                const endPos = markerEnd.getPosition();
                
                if (startPos && endPos) {
                    document.getElementById("start_lat").value = startPos.lat().toFixed(6);
                    document.getElementById("start_lon").value = startPos.lng().toFixed(6);
                    document.getElementById("end_lat").value = endPos.lat().toFixed(6);
                    document.getElementById("end_lon").value = endPos.lng().toFixed(6);
                    
                    updateMapPath();
                }
            } catch (error) {
                console.error("Error updating coordinates:", error);
            }
        }

        // Highlight point on map when hovering chart
        function highlightPointOnMap(lat, lng) {
            try {
                if (!map) return;
                
                const point = new google.maps.LatLng(lat, lng);
                map.setCenter(point);
                
                // Remove previous highlight marker
                if (highlightMarker) {
                    highlightMarker.setMap(null);
                }
                
                // Create new highlight marker
                highlightMarker = new google.maps.Marker({
                    position: point,
                    map: map,
                    icon: {
                        url: "http://maps.google.com/mapfiles/ms/icons/yellow-dot.png"
                    },
                    zIndex: 1000
                });
                
                // Update map info if element exists
                if (mapInfoElement) {
                    mapInfoElement.textContent = `Location: ${lat.toFixed(6)}, ${lng.toFixed(6)}`;
                }
            } catch (error) {
                console.error("Error highlighting point:", error);
            }
        }

        // Initialize the profile chart
        function initProfileChart() {
            try {
                const ctx = document.getElementById('profileCanvas').getContext('2d');
                
                if (profileChart) {
                    profileChart.destroy();
                }
                
                profileChart = new Chart(ctx, {
                    type: 'bar',
                    data: {
                        labels: [],
                        datasets: [
                            {
                                label: 'Terrain Elevation',
                                data: [],
                                backgroundColor: '#10B981',
                                borderColor: '#10B981',
                                borderWidth: 1,
                                barPercentage: 1.0,
                                categoryPercentage: 1.0
                            },
                            {
                                label: 'LOS Path',
                                data: [],
                                borderColor: '#3B82F6',
                                backgroundColor: 'transparent',
                                borderWidth: 2,
                                pointRadius: 0,
                                type: 'line',
                                fill: false
                            },
                            {
                                label: '60% Fresnel Zone (Top)',
                                data: [],
                                backgroundColor: 'rgba(239, 68, 68, 0.2)',
                                borderColor: 'rgba(239, 68, 68, 0.5)',
                                borderWidth: 1,
                                pointRadius: 0,
                                type: 'line',
                                fill: '1'
                            },
                            {
                                label: '60% Fresnel Zone (Bottom)',
                                data: [],
                                backgroundColor: 'rgba(239, 68, 68, 0.2)',
                                borderColor: 'rgba(239, 68, 68, 0.5)',
                                borderWidth: 1,
                                pointRadius: 0,
                                type: 'line',
                                fill: '-1'
                            },
                            {
                                label: 'Full Fresnel Zone (Top)',
                                data: [],
                                backgroundColor: 'rgba(239, 68, 68, 0.1)',
                                borderColor: 'rgba(239, 68, 68, 0.3)',
                                borderWidth: 1,
                                pointRadius: 0,
                                type: 'line',
                                fill: '2'
                            },
                            {
                                label: 'Full Fresnel Zone (Bottom)',
                                data: [],
                                backgroundColor: 'rgba(239, 68, 68, 0.1)',
                                borderColor: 'rgba(239, 68, 68, 0.3)',
                                borderWidth: 1,
                                pointRadius: 0,
                                type: 'line',
                                fill: '-2'
                            }
                        ]
                    },
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        scales: {
                            y: {
                                title: {
                                    display: true,
                                    text: 'Elevation (meters)'
                                },
                                beginAtZero: false
                            },
                            x: {
                                title: {
                                    display: true,
                                    text: 'Distance (km)'
                                }
                            }
                        },
                        plugins: {
                            tooltip: {
                                callbacks: {
                                    label: function(context) {
                                        const pointIndex = context.dataIndex;
                                        const elevation = context.dataset.data[pointIndex];
                                        const distance = distancePoints[pointIndex];
                                        const point = elevationPoints[pointIndex];
                                        
                                        if (point) {
                                            highlightPointOnMap(point.lat, point.lng);
                                        }
                                        
                                        return [
                                            `Distance: ${distance.toFixed(2)} km`,
                                            `Elevation: ${elevation.toFixed(2)} m`
                                        ];
                                    }
                                }
                            },
                            annotation: {
                                annotations: {}
                            }
                        }
                    }
                });
            } catch (error) {
                console.error("Error initializing profile chart:", error);
            }
        }

        function addAntennaMarkers(startTotalHeight, endTotalHeight, distance) {
    try {
        // Remove existing antenna icons
        document.querySelectorAll('.antenna-icon').forEach(el => el.remove());
        
        if (!profileChart || !profileChart.scales || !profileChart.scales.x || !profileChart.scales.y) {
            console.error("Chart not properly initialized");
            return;
        }

        const canvas = document.getElementById('profileCanvas');
        if (!canvas) return;
        
        const rect = canvas.getBoundingClientRect();
        const yScale = profileChart.scales.y;
        const xScale = profileChart.scales.x;
        const chartArea = profileChart.chartArea;

        // Calculate positions with bounds checking
        const startX = chartArea.left;
        let startY = yScale.getPixelForValue(startTotalHeight);
        startY = Math.max(yScale.bottom, Math.min(yScale.top, startY));
        
        const endX = chartArea.right;
        let endY = yScale.getPixelForValue(endTotalHeight);
        endY = Math.max(yScale.bottom, Math.min(yScale.top, endY));

        // Create antenna icons showing total height
        const createAntennaIcon = (x, y, height) => {
            const icon = document.createElement('div');
            icon.className = 'antenna-icon';
            icon.style.left = `${rect.left + x - 12}px`;
            icon.style.top = `${rect.top + y - 24}px`;
            icon.title = `Antenna Height: ${height.toFixed(1)}m (Ground + Antenna)`;
            return icon;
        };
        
        // Add markers
        document.body.appendChild(createAntennaIcon(startX, startY, startTotalHeight));
        document.body.appendChild(createAntennaIcon(endX, endY, endTotalHeight));
    } catch (error) {
        console.error("Error adding antenna markers:", error);
    }
}


       // Update line of sight status display
       function updateLosStatus(hasLos, distance, receivedPower, obstructionPoint) {
            try {
                const losStatus = document.getElementById('losStatus');
                const warningBanner = document.getElementById('warningBanner');
                
                const safeFormat = (value, decimals = 2) => {
                    if (value === undefined || value === null) return 'N/A';
                    const num = typeof value === 'string' ? parseFloat(value) : value;
                    return isNaN(num) ? 'N/A' : num.toFixed(decimals);
                };

                if (hasLos) {
                    warningBanner.classList.add('hidden');
                    losStatus.innerHTML = `
                        <div class="flex items-center text-green-600">
                            <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path>
                            </svg>
                            <span>Clear line of sight (including antenna heights)</span>
                        </div>
                        <div class="mt-2 grid grid-cols-2 gap-4">
                            <div>
                                <p class="text-sm text-gray-600">Distance</p>
                                <p class="font-medium">${safeFormat(distance)} km</p>
                            </div>
                            <div>
                                <p class="text-sm text-gray-600">Received Power</p>
                                <p class="font-medium">${safeFormat(receivedPower)} dBm</p>
                            </div>
                        </div>
                    `;
                } else {
                    warningBanner.classList.remove('hidden');
                    losStatus.innerHTML = `
                        <div class="flex items-center text-red-600">
                            <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                            </svg>
                            <span>Obstruction detected (considering antenna heights)</span>
                        </div>
                        <div class="mt-2 grid grid-cols-2 gap-4">
                            <div>
                                <p class="text-sm text-gray-600">Obstruction Distance</p>
                                <p class="font-medium">${obstructionPoint ? safeFormat(obstructionPoint.distance) : 'N/A'} km</p>
                            </div>
                            <div>
                                <p class="text-sm text-gray-600">Obstruction Height</p>
                                <p class="font-medium">${obstructionPoint ? safeFormat(obstructionPoint.height) : 'N/A'} m</p>
                            </div>
                            <div>
                                <p class="text-sm text-gray-600">Total Distance</p>
                                <p class="font-medium">${safeFormat(distance)} km</p>
                            </div>
                            <div>
                                <p class="text-sm text-gray-600">Received Power</p>
                                <p class="font-medium">${safeFormat(receivedPower)} dBm</p>
                            </div>
                        </div>
                    `;
                }
            } catch (error) {
                console.error("Error updating LOS status:", error);
            }
        }


        function updateProfileChart(data) {
    try {
        if (!profileChart) {
            console.error("Chart not initialized");
            return;
        }

        // Clear previous data
        elevationPoints.length = 0;
        distancePoints.length = 0;
        
        const labels = [];
        const terrainData = [];
        const losPathData = [];
        const f1TopData = [];    // 60% Fresnel top
        const f1BottomData = []; // 60% Fresnel bottom
        const f2TopData = [];    // Full Fresnel top
        const f2BottomData = []; // Full Fresnel bottom
        
        if (data.profile_points && Array.isArray(data.profile_points)) {
            const startTotalHeight = data.start_total_height || (data.start_elevation + (data.start_height || 0));
            const endTotalHeight = data.end_total_height || (data.end_elevation + (data.end_height || 0));
            const totalDistance = data.distance_km || 0;
            
            let hasClearLOS = true;
            let obstructionPoint = null;
            
            // Process all points
            data.profile_points.forEach((point, index) => {
                const distance = point.distance || 0;
                const groundElevation = point.elevation || 0;
                const losHeight = point.los_height || (startTotalHeight + (endTotalHeight - startTotalHeight) * (distance/totalDistance));
                
                labels.push(distance.toFixed(1));
                terrainData.push(groundElevation);
                losPathData.push(losHeight);
                distancePoints.push(distance);
                
                // Calculate Fresnel zones around the LOS path (both above and below)
                f1TopData.push(losHeight + (point.f1_radius || 0)); // 60% zone top
                f1BottomData.push(losHeight - (point.f1_radius || 0)); // 60% zone bottom
                f2TopData.push(losHeight + (point.f2_radius || 0)); // Full zone top
                f2BottomData.push(losHeight - (point.f2_radius || 0)); // Full zone bottom
                
                // Store point data for map interaction
                elevationPoints.push({
                    lat: point.lat || 0,
                    lng: point.lng || 0,
                    groundElevation: groundElevation,
                    distance: distance,
                    losHeight: losHeight,
                    fresnelRadius: point.f2_radius || 0
                });
                
                // Check for obstructions (terrain vs bottom of 60% Fresnel zone)
                const clearanceNeeded = losHeight - (point.f1_radius || 0);
                if (groundElevation > clearanceNeeded) {
                    if (hasClearLOS) {
                        hasClearLOS = false;
                        obstructionPoint = {
                            distance: distance,
                            height: groundElevation - clearanceNeeded,
                            index: index
                        };
                    }
                }
            });

            // Update chart data
            profileChart.data.labels = labels;
            profileChart.data.datasets[0].data = terrainData; // Terrain
            profileChart.data.datasets[1].data = losPathData; // LOS path
            profileChart.data.datasets[2].data = f1TopData;   // 60% top
            profileChart.data.datasets[3].data = f1BottomData; // 60% bottom
            profileChart.data.datasets[4].data = f2TopData;    // Full top
            profileChart.data.datasets[5].data = f2BottomData; // Full bottom
            
            // Adjust Y-axis scale to fit all elements
            const allData = [...terrainData, ...losPathData, ...f1TopData, ...f2TopData];
            const maxValue = Math.max(...allData);
            const minValue = Math.min(0, ...f2BottomData, ...terrainData);
            const range = maxValue - minValue;
            
            profileChart.options.scales.y.min = minValue - range * 0.1;
            profileChart.options.scales.y.max = maxValue + range * 0.1;
            
            // Add antenna markers
            addAntennaMarkers(startTotalHeight, endTotalHeight, totalDistance);
            
            // Update LOS status
            updateLosStatus(hasClearLOS, totalDistance, data.received_power_dBm || 0, obstructionPoint);
            
            profileChart.update();
        }
    } catch (error) {
        console.error("Error updating profile chart:", error);
    }
}

        // Main calculation function
        async function calculateLinkBudget() {
            try {
                const payload = {
                    tx_power: parseFloat(document.getElementById("tx_power").value) || 20,
                    tx_antenna_gain: parseFloat(document.getElementById("tx_antenna_gain").value) || 10,
                    rx_antenna_gain: parseFloat(document.getElementById("rx_antenna_gain").value) || 10,
                    frequency: parseFloat(document.getElementById("frequency").value) || 2.4,
                    start_lat: parseFloat(document.getElementById("start_lat").value) || 37.7749,
                    start_lon: parseFloat(document.getElementById("start_lon").value) || -122.4194,
                    end_lat: parseFloat(document.getElementById("end_lat").value) || 37.7849,
                    end_lon: parseFloat(document.getElementById("end_lon").value) || -122.4094,
                    start_height: parseFloat(document.getElementById("start_height").value) || 10,
                    end_height: parseFloat(document.getElementById("end_height").value) || 10
                };

                // Show loading state
                document.getElementById('losStatus').innerHTML = `
                    <div class="flex items-center text-blue-600">
                        <svg class="animate-spin -ml-1 mr-3 h-5 w-5 text-blue-500" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                            <circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle>
                            <path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                        </svg>
                        <span>Calculating link budget...</span>
                    </div>
                `;

                // Call the FastAPI backend
                const response = await fetch("{{ route('calculate.link') }}", { // Use Laravel's route helper
                                method: "POST",
                                headers: {
                                    "Content-Type": "application/json",
                                    "Accept": "application/json", // Good practice to accept JSON
                                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content') // Add CSRF token
                                },
                                body: JSON.stringify(payload)
                            });
                
                if (!response.ok) {
                    const errorData = await response.json();
                    throw new Error(errorData.detail || "API request failed");
                }
                
                const result = await response.json();
                
                // Update map markers
                if (markerStart && markerEnd) {
                    markerStart.setPosition({ lat: payload.start_lat, lng: payload.start_lon });
                    markerEnd.setPosition({ lat: payload.end_lat, lng: payload.end_lon });
                    updateMapPath();
                }
                
                // Update profile chart with real data
                updateProfileChart(result);

                if (response.status === 422) { // Laravel validation error code
                    const errorData = await response.json();
                    let errorMessages = "Validation failed:\n";
                    if (errorData.errors) {
                        for (const field in errorData.errors) {
                            errorMessages += `- ${errorData.errors[field].join(', ')}\n`;
                        }
                    } else {
                        errorMessages += errorData.message || "Unknown validation error";
                    }
                    throw new Error(errorMessages);
                }
                // Handle other non-OK responses
                if (!response.ok) {
                    const errorData = await response.json();
                    // Use errorData.detail if backend provides it, otherwise use a generic message
                    throw new Error(errorData.detail || `API request failed with status ${response.status}`);
                }

                
            } catch (error) {
                console.error("Error:", error);
                document.getElementById("losStatus").innerHTML = `
                    <div class="text-red-600">
                        <p class="font-medium">Error calculating link budget</p>
                        <p class="text-sm">${error.message}</p>
                    </div>
                `;
            }
        }

        // Initialize everything when page loads
        window.onload = function() {
            try {
                initMap();
            } catch (error) {
                console.error("Initialization error:", error);
                document.getElementById('loadingOverlay').innerHTML = `
                    <div class="text-center">
                        <div class="text-red-500 mb-4">
                            <svg class="w-12 h-12 mx-auto" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"></path>
                            </svg>
                        </div>
                        <p class="text-red-600 font-medium">Failed to initialize application</p>
                        <p class="text-gray-600 mt-2">${error.message}</p>
                        <button onclick="window.location.reload()" class="mt-4 px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700">
                            Reload Page
                        </button>
                    </div>
                `;
            }
        };
    </script>
</body>
</html>